<?php

namespace App\Filament\Pages;

use Filament\Pages\Page;
use Filament\Notifications\Notification;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Form;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Radio;
use Filament\Forms\Get;
use App\Models\Setting;

class Settings extends Page implements HasForms
{
    use InteractsWithForms;

    protected static ?string $navigationIcon = 'heroicon-o-cog-6-tooth';
    protected static ?string $navigationLabel = 'Pengaturan Toko';
    protected static ?string $navigationGroup = 'Pengaturan Toko';
    protected static ?string $slug = 'shop-settings';
    protected static ?int $navigationSort = 9;
    protected static string $view = 'filament.pages.settings';

    public ?array $data = [];

    public function mount(): void
    {
        $setting = Setting::first();
        $this->form->fill($setting ? $setting->toArray() : []);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Profil Toko')
                    ->description('Informasi dasar toko Anda')
                    ->schema([
                        TextInput::make('name')
                            ->label('Nama Toko')
                            ->required()
                            ->maxLength(255),

                        TextInput::make('address')
                            ->label('Alamat Toko')
                            ->required()
                            ->maxLength(255),

                        TextInput::make('phone')
                            ->label('Nomor Telepon')
                            ->tel()
                            ->required()
                            ->maxLength(255),

                        FileUpload::make('logo')
                            ->label('Logo Toko')
                            ->disk('public')
                            ->image()
                            ->directory('logos')
                            ->acceptedFileTypes(['image/png'])
                            ->helperText('Upload logo toko dalam format PNG')
                            ->columnSpanFull(),
                    ])
                    ->columns(2),

                Section::make('Pengaturan Printer')
                    ->description('Konfigurasi printer untuk struk')
                    ->schema([
                        Radio::make('print_via_bluetooth')
                            ->label('Tipe Koneksi Printer')
                            ->options([
                                '0' => 'Kabel (Server Local)',
                                '1' => 'Bluetooth'
                            ])
                            ->inline()
                            ->default('0')
                            ->live()
                            ->helperText('Pilih metode koneksi printer'),

                        TextInput::make('name_printer_local')
                            ->label('Nama Printer')
                            ->maxLength(255)
                            ->helperText('Contoh: Epson T20, Canon PIXMA, dll')
                            ->visible(fn (Get $get): bool => $get('print_via_bluetooth') === '0'),
                    ])
                    ->columns(1),
            ])
            ->statePath('data');
    }

    public function save(): void
    {
        $data = $this->form->getState();

        Setting::updateOrCreate([], $data);

        Notification::make()
            ->success()
            ->title('Pengaturan Disimpan')
            ->body('Pengaturan toko berhasil disimpan.')
            ->send();
    }


}
