<?php

namespace App\Livewire;

use Filament\Forms;
use App\Models\User;
use App\Models\Product;
use App\Models\Setting;
use Livewire\Component;
use App\Models\Category;
use Filament\Forms\Form;
use App\Models\Transaction;
use Livewire\WithPagination;
use App\Models\PaymentMethod;
use App\Models\TransactionItem;
use App\Helpers\TransactionHelper;
use App\Helpers\PromoHelper;
use App\Services\DirectPrintService;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;

class Pos extends Component
{
    use WithPagination {
        gotoPage as protected traitGotoPage;
        previousPage as protected traitPreviousPage;
        nextPage as protected traitNextPage;
    }

    public int | string $perPage = 10;
    public $categories;
    public $selectedCategory;
    public $search = '';
    public $print_via_bluetooth = false;
    public $barcode = '';
    public $name = 'Umum';
    public $payment_method_id;
    public $payment_methods;
    public $order_items = [];
    public $total_price = 0;
    public $promo_discount = 0;
    public $applied_promos = [];
    public $cash_received = '';
    public $change = 0;
    public $showConfirmationModal = false;
    public $showCheckoutModal = false;
    public $orderToPrint = null;
    public $is_cash = true;
    public $selected_payment_method = null;
    public bool $showWeightModal = false;
    public $weight_gram = '';
    public ?int $weightProductId = null;

    protected $listeners = [
        'scanResult' => 'handleScanResult',
    ];

    public function mount()
    {
        $settings = Setting::first();
        $this->print_via_bluetooth = $settings->print_via_bluetooth ?? $this->print_via_bluetooth = false;

        // Mengambil data kategori dan menambahkan data 'Semua' sebagai pilihan pertama
        $this->categories = collect([['id' => null, 'name' => 'Semua']])->merge(Category::all());

        // Jika session 'orderItems' ada, maka ambil data nya dan simpan ke dalam property $order_items
        // Session 'orderItems' digunakan untuk menyimpan data order sementara sebelum di checkout
        if (session()->has('orderItems')) {
            $this->order_items = session('orderItems');
            $this->calculateTotal();
        }

        $this->payment_methods = PaymentMethod::all();
    }

    public function render()
    {
        return view('livewire.pos', [
            'products' => Product::where(function($query) {
                    // Product dengan stok reguler atau stok kongsi yang > 0
                    $query->where('stock', '>', 0)
                          ->orWhere('stok_kongsi', '>', 0);
                })
                ->where('is_active', 1)
                ->when($this->selectedCategory !== null, function ($query) {
                    return $query->where('category_id', $this->selectedCategory);
                })
                ->where(function ($query) {
                    return $query->where('name', 'LIKE', '%' . $this->search . '%')
                        ->orWhere('sku', 'LIKE', '%' . $this->search . '%');
                })
                ->paginate($this->perPage)
        ]);
    }


    public function updatedPaymentMethodId($value)
    {
        if ($value) {
            // Recalculate total terlebih dahulu untuk memastikan promo discount terhitung
            $this->calculateTotal();

            $paymentMethod = PaymentMethod::find($value);
            $this->selected_payment_method = $paymentMethod;
            $this->is_cash = $paymentMethod->is_cash ?? false;

            if (!$this->is_cash) {
                $this->cash_received = $this->total_price;
                $this->change = 0;
            } else {
                $this->calculateChange();
            }
        }
    }

    public function updatedCashReceived($value)
    {
        if ($this->is_cash) {
            // Remove thousand separator dots before calculation
            $this->cash_received = $value;
            $this->calculateChange();
        }
    }

    public function calculateChange()
    {
        // Remove thousand separator dots and convert to number
        $cleanValue = str_replace('.', '', $this->cash_received);
        $cashReceived = floatval($cleanValue);
        $totalPrice = floatval($this->total_price);

        if ($cashReceived >= $totalPrice) {
            $this->change = $cashReceived - $totalPrice;
        } else {
            $this->change = 0;
        }
    }

    // Helper method to get numeric value from formatted input
    public function getCashReceivedNumeric()
    {
        return floatval(str_replace('.', '', $this->cash_received));
    }

    public function updatedBarcode($barcode)
    {
        $product = Product::where('barcode', $barcode)
            ->where('is_active', true)->first();

        if ($product) {
            $this->addToOrder($product->id);
        } else {
            Notification::make()
                ->title('Product not found ' . $barcode)
                ->danger()
                ->send();
            // Focus back to barcode input even if product not found
            $this->dispatch('focus-barcode');
        }

        // Reset barcode
        $this->barcode = '';
    }

    public function handleScanResult($decodedText)
    {
        $product = Product::where('barcode', $decodedText)
            ->where('is_active', true)->first();

        if ($product) {
            $this->addToOrder($product->id);
        } else {
            Notification::make()
                ->title('Product not found ' . $decodedText)
                ->danger()
                ->send();
            // Focus back to barcode input even if product not found
            $this->dispatch('focus-barcode');
        }

        // Reset barcode
        $this->barcode = '';
    }

    public function setCategory($categoryId = null)
    {
        $this->selectedCategory = $categoryId;
        // Reset pagination page when category changes to ensure correct results
        $this->resetPage();
    }

    /**
     * Reset pagination to page 1 when search term changes
     */
    public function updatingSearch()
    {
        $this->resetPage();
    }

    /**
     * Reset pagination to page 1 when selected category changes (for direct property updates)
     */
    public function updatingSelectedCategory()
    {
        $this->resetPage();
    }

    public function addToOrder($productId)
    {
        $product = Product::find($productId);

        if ($product) {
            // If product sold by weight, prompt for weight input
            if (!empty($product->price_per_kg)) {
                $this->weightProductId = $product->id;
                $this->showWeightModal = true;
                return;
            }
            // Cari apakah item sudah ada di dalam order
            $existingItemKey = array_search($productId, array_column($this->order_items, 'product_id'));

            // Jika item sudah ada, tambahkan 1 quantity
            if ($existingItemKey !== false) {
                $totalAvailableStock = $product->getTotalAvailableStock();
                if ($this->order_items[$existingItemKey]['quantity'] >= $totalAvailableStock) {
                    Notification::make()
                        ->title('Stok barang tidak mencukupi')
                        ->body("Stok tersedia: {$product->getFormattedStock()}")
                        ->danger()
                        ->send();
                    return;
                } else {
                    $this->order_items[$existingItemKey]['quantity']++;
                }
            }
            // Jika item belum ada, tambahkan item baru ke dalam order
            else {
                // Get final price (with discount if active)
                $finalPrice = $product->getFinalPrice();

                $this->order_items[] = [
                    'product_id' => $product->id,
                    'name' => $product->name,
                    'price' => $finalPrice, // Use discounted price
                    'original_price' => $product->price, // Store original price for reference
                    'discount_percentage' => $product->hasActiveDiscount() ? $product->discount_percentage : 0,
                    'cost_price' => $product->cost_price,
                    'total_profit' => $finalPrice - $product->cost_price,
                    'image_url' => $product->image,
                    'quantity' => 1,
                ];
            }

            // Simpan perubahan order ke session
            session()->put('orderItems', $this->order_items);

            // Recalculate total
            $this->calculateTotal();

            // Recalculate change if cash payment
            if ($this->is_cash && !empty($this->cash_received)) {
                $this->calculateChange();
            }

            // Focus back to barcode input
            $this->dispatch('focus-barcode');
        }
    }

    public function confirmWeight()
    {
        // Find product and calculate price by weight
        $product = Product::find($this->weightProductId);
        $grams = floatval($this->weight_gram);
        $kilograms = $grams / 1000;

        // Check if stock is sufficient
        $totalAvailableStock = $product->getTotalAvailableStock();
        if ($kilograms > $totalAvailableStock) {
            Notification::make()
                ->title('Stok barang tidak mencukupi')
                ->body("Stok tersedia: {$product->getFormattedStock()}, Anda minta: " . number_format($kilograms, 3, ',', '.') . " kg")
                ->danger()
                ->send();
            return;
        }

        // Use discounted price per kg if available
        $unitPrice = $product->getFinalPricePerKg() ?? $product->price_per_kg;
        $calculatedPrice = round($unitPrice * $kilograms);

        // Calculate proportional cost price based on weight
        $costPricePerKg = $product->cost_price ?? 0;
        $proportionalCostPrice = round($costPricePerKg * $kilograms);

        // Add item to order with quantity 1, price based on weight
        $this->order_items[] = [
            'product_id' => $product->id,
            'name' => $product->name . " ({$grams}gr)",
            'price' => $calculatedPrice,
            'original_price' => round($product->price_per_kg * $kilograms), // Store original price for reference
            'discount_percentage' => $product->hasActiveDiscount() ? $product->discount_percentage : 0,
            'cost_price' => $proportionalCostPrice, // Use proportional cost price
            'total_profit' => $calculatedPrice - $proportionalCostPrice, // Calculate profit based on proportional cost
            'image_url' => $product->image,
            'quantity' => 1,
            'weight_grams' => $grams, // Store weight for display
            'weight_kg' => $kilograms, // Store weight in kg for stock calculation
            'unit_price' => $unitPrice, // Store discounted unit price for display
            'original_unit_price' => $product->price_per_kg, // Store original unit price
        ];
        // Reset modal state and recalculate
        $this->showWeightModal = false;
        $this->weight_gram = '';
        session()->put('orderItems', $this->order_items);
        $this->calculateTotal();
        if ($this->is_cash && !empty($this->cash_received)) {
            $this->calculateChange();
        }
    }

    public function loadOrderItems($orderItems)
    {
        $this->order_items = $orderItems;
        session()->put('orderItems', $orderItems);
    }

    public function increaseQuantity($product_id)
    {
        $product = Product::find($product_id);

        if (!$product) {
            Notification::make()
                ->title('Produk tidak ditemukan')
                ->danger()
                ->send();
            return;
        }

        // Loop setiap item yang ada di cart
        foreach ($this->order_items as $key => $item) {
            // Jika item yang sedang di-loop sama dengan item yang ingin di tambah
            if ($item['product_id'] == $product_id) {
                // Jika quantity item ditambah 1 masih kurang dari atau sama dengan total stok tersedia maka tambah 1 quantity
                $totalAvailableStock = $product->getTotalAvailableStock();
                if ($item['quantity'] + 1 <= $totalAvailableStock) {
                    $this->order_items[$key]['quantity']++;
                }
                // Jika quantity item yang ingin di tambah lebih besar dari total stok tersedia maka tampilkan notifikasi
                else {
                    Notification::make()
                        ->title('Stok barang tidak mencukupi')
                        ->body("Stok tersedia: {$product->getFormattedStock()}")
                        ->danger()
                        ->send();
                }
                // Berhenti loop karena item yang ingin di tambah sudah di temukan
                break;
            }
        }

        session()->put('orderItems', $this->order_items);

        // Recalculate total and change
        $this->calculateTotal();
        if ($this->is_cash && !empty($this->cash_received)) {
            $this->calculateChange();
        }
    }

    public function decreaseQuantity($product_id)
    {
        // Loop setiap item yang ada di cart
        foreach ($this->order_items as $key => $item) {
            // Jika item yang sedang di-loop sama dengan item yang ingin di kurangi
            if ($item['product_id'] == $product_id) {
                // Jika quantity item lebih dari 1 maka kurangi 1 quantity
                if ($this->order_items[$key]['quantity'] > 1) {
                    $this->order_items[$key]['quantity']--;
                }
                // Jika quantity item 1 maka hapus item dari cart
                else {
                    unset($this->order_items[$key]);
                    $this->order_items = array_values($this->order_items);
                }
                break;
            }
        }

        // Simpan perubahan cart ke session
        session()->put('orderItems', $this->order_items);

        // Recalculate total and change
        $this->calculateTotal();
        if ($this->is_cash && !empty($this->cash_received)) {
            $this->calculateChange();
        }
    }

    /**
     * Remove an item entirely from the cart.
     */
    public function removeItem($product_id)
    {
        foreach ($this->order_items as $key => $item) {
            if ($item['product_id'] == $product_id) {
                unset($this->order_items[$key]);
                $this->order_items = array_values($this->order_items);
                break;
            }
        }
        session()->put('orderItems', $this->order_items);
        $this->calculateTotal();
        if ($this->is_cash && !empty($this->cash_received)) {
            $this->calculateChange();
        }
    }

    public function calculateTotal()
    {
        // Inisialisasi total harga
        $total = 0;

        // Loop setiap item yang ada di cart
        foreach ($this->order_items as $item) {
            // Tambahkan harga setiap item ke total
            $total += $item['quantity'] * $item['price'];
        }

        // Calculate promo discounts
        $promoResult = PromoHelper::calculatePromos($this->order_items);
        $this->applied_promos = $promoResult['promos'];
        $this->promo_discount = $promoResult['total_discount'];

        // Simpan total harga sebelum diskon
        $this->total_price = max(0, $total - $this->promo_discount);

        // Simpan ke session untuk customer display
        session()->put('totalPrice', $total);
        session()->put('promoDiscount', $this->promo_discount);
        session()->put('transactionName', $this->name);

        // Return total harga
        return $this->total_price;
    }

    public function resetOrder()
    {
        // Hapus semua session terkait
        session()->forget(['orderItems', 'name', 'payment_method_id', 'totalPrice', 'promoDiscount', 'transactionName']);

        // Reset variabel Livewire
        $this->order_items = [];
        $this->payment_method_id = null;
        $this->total_price = 0;
        $this->cash_received = '';
        $this->change = 0;
        $this->is_cash = true;
        $this->selected_payment_method = null;
    }

    public function formatNumber($value)
    {
        return number_format($value, 0, ',', '.');
    }

    public function checkout()
    {
        // Convert formatted cash_received to numeric for validation
        $cashReceivedNumeric = $this->getCashReceivedNumeric();

        // Custom validation messages
        $messages = [
            'payment_method_id.required' => 'Metode pembayaran harus dipilih',
            'cash_received.required' => 'Nominal bayar harus diisi',
            'cash_received_numeric.min' => 'Nominal bayar kurang dari total belanja'
        ];

        // Base validation
        $this->validate([
            'name' => 'string|max:255',
            'payment_method_id' => 'required'
        ], $messages);

        // Additional validation for cash payment
        if ($this->is_cash) {
            if (empty($this->cash_received)) {
                $this->addError('cash_received', 'Nominal bayar harus diisi');
                return;
            }

            if ($cashReceivedNumeric < $this->total_price) {
                $this->addError('cash_received', 'Nominal bayar kurang dari total belanja');
                return;
            }
        }

        $payment_method_id_temp = $this->payment_method_id;
        $cash_received_value = $this->is_cash ? $cashReceivedNumeric : $this->total_price;

        if (session('orderItems') === null || count(session('orderItems')) == 0) {
            Notification::make()
                ->title('Keranjang kosong')
                ->danger()
                ->send();

            $this->showCheckoutModal = false;
        } else {
            $order = Transaction::create([
                'payment_method_id' => $payment_method_id_temp,
                'transaction_number' => TransactionHelper::generateUniqueTrxId(),
                'name' => $this->name,
                'total' => $this->total_price,
                'cash_received' => $cash_received_value,
                'change' => $this->change,
                'promo_discount' => $this->promo_discount,
            ]);

            foreach ($this->order_items as $item) {
                TransactionItem::create([
                    'transaction_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'cost_price' => $item['cost_price'],
                    'total_profit' => $item['total_profit'] * $item['quantity'],
                    'weight' => isset($item['weight_kg']) ? $item['weight_kg'] : (isset($item['weight_grams']) ? $item['weight_grams'] / 1000 : null),
                ]);
            }

            // Simpan ID order untuk cetak
            $this->orderToPrint = $order->id;

            // Siapkan data struk untuk customer display
            $receiptData = [
                'transaction_number' => $order->transaction_number,
                'date' => $order->created_at->format('d-m-Y H:i:s'),
                'items' => $this->order_items,
                'subtotal' => $this->total_price + $this->promo_discount,
                'promo_discount' => $this->promo_discount,
                'total' => $this->total_price,
                'cash_received' => $cash_received_value,
                'change' => $this->change,
            ];

            // Simpan ke session untuk customer display
            session()->put('completedTransaction', $receiptData);

            // Tampilkan modal konfirmasi
            $this->showConfirmationModal = true;
            $this->showCheckoutModal = false;

            Notification::make()
                ->title('Order berhasil disimpan')
                ->success()
                ->send();

            $this->resetOrder();
        }
    }

    public function printLocalKabel()
    {
        $directPrint = app(DirectPrintService::class);

        $directPrint->print($this->orderToPrint);

        $this->showConfirmationModal = false;
        $this->orderToPrint = null;
    }

    public function printBluetooth()
    {
        $order = Transaction::with(['paymentMethod', 'transactionItems.product'])->findOrFail($this->orderToPrint);
        $items = $order->transactionItems;

        $this->dispatch(
            'doPrintReceipt',
            store: Setting::first(),
            order: $order,
            items: $items,
            date: $order->created_at->format('d-m-Y H:i:s'),
            cashier: Auth::user()
        );

        $this->showConfirmationModal = false;
        $this->orderToPrint = null;
    }

    // --- Temporary debug overrides for pagination ---
    // These will show a Filament notification when pagination methods are invoked.
    // Remove these after debugging.
    public function previousPage($pageName = 'page')
    {
        // Call original WithPagination implementation
        $this->traitPreviousPage($pageName);
        // Visual and browser-side debug events
        Notification::make()->title('Pagination: previousPage called')->success()->send();
        $this->dispatch('pagination-called', method: 'previousPage', pageName: $pageName);
    }

    public function nextPage($pageName = 'page')
    {
        $this->traitNextPage($pageName);
        Notification::make()->title('Pagination: nextPage called')->success()->send();
        $this->dispatch('pagination-called', method: 'nextPage', pageName: $pageName);
    }

    public function gotoPage($page, $pageName = 'page')
    {
        $this->traitGotoPage($page, $pageName);
        Notification::make()->title("Pagination: gotoPage({$page}) called")->success()->send();
        $this->dispatch('pagination-called', method: 'gotoPage', page: $page, pageName: $pageName);
    }
}

